unit keystuff;

{$B+,O-}

{----------------------------------------------------------------------}
{------  Turbo Pascal KEYSTUFF unit written by Douglas Webb    --------}
{------  Copywrite (c) 1989,1990.  All rights reserved.        --------}
{------  Last undated 08/23/90                                 --------}
{----------------------------------------------------------------------}
{------  This sofware is public domain, and no fee should be   --------}
{------   charged for it's distribution.  It may not be        --------}
{------   redistibuted it in any altered format nor may this   --------}
{------   notice be removed.                                   --------}
{----------------------------------------------------------------------}
{------  DISCLAIMER:  There shall be no guarantee of the       --------}
{------   suitability of this software for any purpose.  The   --------}
{------   author shall not be liable for any damages arrising  --------}
{------   from the use of this software.                       --------}
{----------------------------------------------------------------------}



interface
uses CRT,DOS;

CONST

{
   The following are definitions for a large number of extended keys
   which are commonly used.
   * is used to indicate key codes which are not normally returned by
   the BIOS, ie. using these keys normally has no effect at all.  You
   need the table below to decode them if you get these values, which will
   happen if the 'ExtendKeyboard' interrupt is being used to augment the
   BIOS's keyboard handling functions.
}

  NoKey = #242;   { Is returned by 'Nextkey' if no key is ready. }


  NULL : Char = #0;
  BackSpace = #8;
  TAB       = #9;
  Linefeed  = #10;
  Return    = #13;
  ESC       = #27;
  Space     = #32;
  SHIFT_TAB = #143;


  CTRL_A = #1;            ALT_A = #158;
  CTRL_B = #2;            ALT_B = #176;
  CTRL_C = #3;            ALT_C = #174;
  CTRL_D = #4;            ALT_D = #160;
  CTRL_E = #5;            ALT_E = #146;
  CTRL_F = #6;            ALT_F = #161;
  CTRL_G = #7;{Bell}      ALT_G = #162;
  CTRL_H = #8;{Bspace}    ALT_H = #163;
  CTRL_I = #9;{TAB}       ALT_I = #151;
  CTRL_J = #10;{Linefeed} ALT_J = #164;
  CTRL_K = #11;           ALT_K = #165;
  CTRL_L = #12;{Pg Eject} ALT_L = #166;
  CTRL_M = #13;{C return} ALT_M = #178;
  CTRL_N = #14;           ALT_N = #177;
  CTRL_O = #15;           ALT_O = #152;
  CTRL_P = #16;           ALT_P = #153;
  CTRL_Q = #17;           ALT_Q = #144;
  CTRL_R = #18;           ALT_R = #147;
  CTRL_S = #19;           ALT_S = #159;
  CTRL_T = #20;           ALT_T = #148;
  CTRL_U = #21;           ALT_U = #150;
  CTRL_V = #22;           ALT_V = #175;
  CTRL_W = #23;           ALT_W = #145;
  CTRL_X = #24;           ALT_X = #173;
  CTRL_Y = #25;           ALT_Y = #149;
  CTRL_Z = #26;{EOF}      ALT_Z = #172;


  F1     = #187;    SHIFT_F1  = #212;     CTRL_F1  = #222;    ALT_F1  = #232;
  F2     = #188;    SHIFT_F2  = #213;     CTRL_F2  = #223;    ALT_F2  = #233;
  F3     = #189;    SHIFT_F3  = #214;     CTRL_F3  = #224;    ALT_F3  = #234;
  F4     = #190;    SHIFT_F4  = #215;     CTRL_F4  = #225;    ALT_F4  = #235;
  F5     = #191;    SHIFT_F5  = #216;     CTRL_F5  = #226;    ALT_F5  = #236;
  F6     = #192;    SHIFT_F6  = #217;     CTRL_F6  = #227;    ALT_F6  = #237;
  F7     = #193;    SHIFT_F7  = #218;     CTRL_F7  = #228;    ALT_F7  = #238;
  F8     = #194;    SHIFT_F8  = #219;     CTRL_F8  = #229;    ALT_F8  = #239;
  F9     = #195;    SHIFT_F9  = #220;     CTRL_F9  = #230;    ALT_F9  = #240;
  F10    = #196;    SHIFT_F10 = #221;     CTRL_F10 = #231;    ALT_F10 = #241;

{  ----------------  Optional extended keyboard support -----------------}
  F11    = #133;    SHIFT_F11 = #135;     CRTL_F11 = #137;    ALT_F11 = #139;
  F12    = #134;    SHIFT_F12 = #136;     CRTL_F12 = #138;    ALT_F12 = #140;


  Home   = #199;    CTRL_Home  = #247;      ALT_Home  = #179;{*}
  UP     = #200;    CTRL_UP    = #141;{*}   ALT_UP    = #180;{*}
  PGUP   = #201;    CTRL_PGUP  = #132;      ALT_PGUP  = #181;{*}
  Left   = #203;    CTRL_Left  = #243;      ALT_Left  = #182;{*}
  Five   = #204;{*} CTRL_Five  = #142;{*}   ALT_Five  = #183;{*}
  Right  = #205;    CTRL_Right = #244;      ALT_Right = #184;{*}
  EndKey = #207;    CTRL_End   = #245;      ALT_End   = #185;{*}
  Down   = #208;    CTRL_Down  = #154;{*}   ALT_Down  = #186;{*}
  PGDN   = #209;    CTRL_PGDN  = #246;      ALT_PGDN  = #197;{*}
  INS    = #210;    CTRL_INS   = #155;{*}   ALT_INS   = #198;{*}
  DEL    = #211;    CTRL_DEL   = #156;{*}   ALT_DEL   = #206;{*}


Procedure Stuffkey(Key: Char);   { Push a key into the keyboard buffer. }
Procedure Extend_KeyBoard;
    { Extends BIOS to allows keys marked with * to be recognized. }
Function AutoType(AString : STRING): WORD;
    { Send a string into the keyboard buffer. }
Function NextKey(VAR AuxCode:CHAR):CHAR;
    { Returns the next key, if one exists, returns constant 'NoKey' otherwise. }
Function GetKey:Char;
    { Gets next key as a 1 char value, even function keys and ALT-key combinations. }
Procedure ClearKBDBuff; {Clears the keyboardbuffer. }
Procedure CursorON;     { Turns cursor on.  }
Procedure CursorOFF;    { Turns cursor off. }


implementation


CONST

  KeyBoardPort : BYTE = $60;  { Hardware location for keyboard scan codes. }

  ABuffer_Size = 256;
  AutoTyping : BOOLEAN = FALSE;
  ABuffer_Head : WORD = 0;
  ABuffer_Tail : WORD = 0;



VAR
  Head : WORD absolute $40:$1A;
  Tail : WORD absolute $40:$1C;
  ShiftKeys : BYTE absolute $40:$17;  { Byte containing shift key status. }


  BIOS_INT9 : POINTER;    { Pointer to the hardware keyboard interrupt. }
  BIOS_INT16 : POINTER;   { Pointer to the software keyboard interrupt. }
  Exit_Vec,Exit_Vec2  : POINTER;    { Pointer previous exit vector.      }



  ABuffer : Array[0..PRED(ABuffer_Size)] OF CHAR;


  ch : Char;





PROCEDURE EnableInterrupts; INLINE($FB);
PROCEDURE DisableInterrupts; INLINE($FA);


PROCEDURE CallInterrupt(oldvector : Pointer); { stack image     }
   INLINE($55/          { PUSH    BP                 } {  ip   \ return  }
          $89/$E5/      { MOV     BP,SP              } {  cs     to here }
          $9C/          { PUSHF create an IRET return} {  flags/         }
          $36/          { SS:                        } {  bp  <--sp      }
          $FF/$5E/$02/  { CALLfar [BP+02]            } {  cs \           }
          $5D/          { POP     BP                 } {  ip /old vector }
          $83/$C4/$04); { ADD     SP,+04             } {                 }
   {END CallInterrupt}


PROCEDURE JumpToInterrupt(oldvector : Pointer);
INLINE(                        { Jump to old Intr from local ISR }
   $5B/                        { POP  BX IP part of vector     }
   $58/                        { POP  AX CS part of vector     }
   $87/$5E/$0E/                { XCHG BX,[BP+14] switch ofs/bx }
   $87/$46/$10/                { XCHG AX,[BP+16] switch seg/ax }
   $8B/$E5/                    { MOV  SP,BP                    }
   $5D/                        { POP  BP                       }
   $07/                        { POP  ES                       }
   $1F/                        { POP  DS                       }
   $5F/                        { POP  DI                       }
   $5E/                        { POP  SI                       }
   $5A/                        { POP  DX                       }
   $59/                        { POP  CX                       }
   $CB                         { RETF      Jump [ToOldVector]  }
  );
   {end JumpToInterrupt}




Procedure Stuffkey(Key: Char);

   { This function stuffs the character passed as a parameter into the
       keyboard buffer.  Thereafter use of 'Readkey' or this unit's 'Getkey'
       or 'Next key functions will return this key value.
   }

VAR
  KeyV : WORD;

BEGIN
  KeyV := ORD(Key);
  IF (KeyV AND 128) = 128 THEN       { Is it a special key. }
    KeyV := (KeyV SHL 8) AND $7F00;
  IF ((Tail+2 = Head) OR ((Tail = $3C) AND (Head = $1E))) THEN  { Buffer Full. }
    Exit;                            { Abort buffer full. }
  MemW[$40:Tail] := KeyV;
  INC(Tail,2);
  IF Tail = $3E THEN Tail := $1E;
END;






{--------------------------------------------------------------------------}
{    The following 3 procedure implement an interrupt service routine      }
{   extends the keys that put a value into the keyboard buffer. All keys   }
{   listed with a * above are implemented.  Calling 'Extend_KeyBoard'      }
{   installs the interrupt handler.                                        }
{--------------------------------------------------------------------------}


Procedure ExtendKeyBoardISR(_Flags,_CS,_IP,_AX,_BX,_CX,_DX,_SI,_DI,_DS,_ES,_BP:WORD);
Interrupt;

  { This is interrupt handler, which when installed by a call to
     "Extend_KeyBoard" will subplement the existing BIOS interrupt 9
     which interprets keystrokes and puts the results in the keyboard
     buffer.  Some keys (those indicated with a '*' in the list of constants
     above normally produce no key in the keyboard buffer.  When installed
     the value above will be found in the keyboard buffer, ie. if ALT-DEL
     is pressed the value 206 will be the next value placed in the keyboard
     buffer.  Unlike normal extended keys, no 0, # combination is placed in
     the buffer, just the single value indicated above.  This will allow
     "Readkey" to read many more key combinations.
  }

TYPE
  Keypad_Xlate_Type = Array[1..17] OF RECORD
                                         Code     : CHAR;
                                         ShiftKey : BYTE;
                                         ScanCode: BYTE;
                                       END;
CONST
  KbdXLate : KeyPad_XLate_Type =
                    ((Code : FIVE;      ShiftKey : 0; ScanCode : 76),
                     (Code : CTRL_UP;   ShiftKey : 4; ScanCode : 72),
                     (Code : CTRL_Five; ShiftKey : 4; ScanCode : 76),
                     (Code : CTRL_Down; ShiftKey : 4; ScanCode : 80),
                     (Code : CTRL_INS;  ShiftKey : 4; ScanCode : 82),
                     (Code : CTRL_DEL;  ShiftKey : 4; ScanCode : 83),
                     (Code : ALT_Home;  ShiftKey : 8; ScanCode : 71),
                     (Code : ALT_UP;    ShiftKey : 8; ScanCode : 72),
                     (Code : ALT_PGUP;  ShiftKey : 8; ScanCode : 73),
                     (Code : ALT_Left;  ShiftKey : 8; ScanCode : 75),
                     (Code : ALT_Five;  ShiftKey : 8; ScanCode : 76),
                     (Code : ALT_Right; ShiftKey : 8; ScanCode : 77),
                     (Code : ALT_End;   ShiftKey : 8; ScanCode : 79),
                     (Code : ALT_Down;  ShiftKey : 8; ScanCode : 80),
                     (Code : ALT_PGDN;  ShiftKey : 8; ScanCode : 81),
                     (Code : ALT_INS;   ShiftKey : 8; ScanCode : 82),
                     (Code : ALT_DEL;   ShiftKey : 8; ScanCode : 83));

VAR
  SCode,Shiftstatus,X : BYTE;
  XKey :Boolean;

BEGIN
  Enableinterrupts;
  XKey := FALSE;
  SCode := port[KeyBoardPort];
  ShiftStatus := ShiftKeys AND $F;  { Mask Off Insert status,NUMlock,Scrolllock }
  IF (SCode < 71) OR       { If not in our scan code range OR key release. }
    (SCode > $7F) THEN Jumptointerrupt(BIOS_INT9);
  FOR X := 1 TO 17 DO
    IF (KbdXLate[X].ScanCode = SCode) AND (KbdXLate[x].ShiftKey = ShiftStatus) THEN
      BEGIN
        Stuffkey(KbdXLate[X].Code);
        XKey := TRUE;
      END;
  IF XKey THEN        { Tell keyboard controller that key was handled. }
    INLINE($E4/$61/       {IN   AL,$61     ;Read Kbd controller Port}
           $88/$C4/       {MOV  AH,AL      ;                        }
           $0C/$80/       {OR   AL,$80     ;Set the "reset" bit and }
           $E6/$61/       {OUT  $61,AL     ; send it back to control}
           $86/$C4/       {XCHG AH,AL      ;Get back control value  }
           $E6/$61/       {OUT  $61,AL     ; and send it too.       }
           $FA/           {CLI             ;No interrupts           }
           $B0/$20/       {MOV  AL,$20     ;Send an EOI to the      }
           $E6/$20)       {OUT  $20,AL     ; interrupt controller.  }
  ELSE
    JumpToInterrupt(BIOS_INT9);
END;




{$F+} Procedure Critical_Exit; {$F-}
  { Close up, the program is ending. }
BEGIN
  ExitProc := Exit_Vec;          { Reset exit vector. }
  DisableInterrupts;
  SetIntVec(9, BIOS_INT9); { Restore KeyBoard Hardware ISR. }
  EnableInterrupts;
END;



Procedure Extend_KeyBoard;
BEGIN
   Exit_Vec := ExitProc;          { Chain into ExitProc     }
   ExitProc := @Critical_Exit;    { install additional exit }
   DisableInterrupts;
   GetIntVec(9, BIOS_INT9);        { Save old keyboard hardware ISR. }
   SetIntVec(9, @ExtendKeyBoardISR); { Set New KeyBoard hardware ISR. }
   Enableinterrupts;
END;







{--------------------------------------------------------------------------}
{    The following 3 procedure implement an interrupt service routine      }
{   extends the functionality of the function stuffkey above.  Because the }
{   keyboard buffer can hold only 15 characters at a time some more        }
{   comprehensive systems is needed if for example a large macro is to be  }
{   executed.  'Autotype' allows you to put up to 256 characters at a time }
{   into a buffer which will then stuff the keyboard buffer, as the keys   }
{   are read.                                                              }
{--------------------------------------------------------------------------}


Procedure AutoTypeISR(Flags,CS,IP,AX,BX,CX,DX,SI,DI,DS,ES,BP:WORD);
Interrupt;

CONST
  MaskZero = $FFBF;    { Mask to zero the zero flag. }

VAR
  Service : BYTE;

BEGIN
  IF (ABuffer_Head = ABuffer_Tail) OR (Head <> Tail) THEN
    JumpToInterrupt(BIOS_INT16);
  Service := Hi(AX);
  IF Service IN [0,1] THEN
    BEGIN
      Stuffkey(ABuffer[ABuffer_Tail]);
      ABuffer_Tail := SUCC(ABuffer_Tail) MOD ABuffer_SIze;
    END;
  JumpToInterrupt(BIOS_INT16);
END;




{$F+} Procedure Critical_Exit2; {$F-}
  { Close up, the program is ending. }
BEGIN
  ExitProc := Exit_Vec2;          { Reset exit vector. }
  DisableInterrupts;
  SetIntVec(9, BIOS_INT16); { Restore KeyBoard Software ISR. }
  EnableInterrupts;
END;


Function AutoType(AString : STRING): WORD;
  { Puts a string in the autotype buffer, returns number of characters
     successfully placed in buffer. Buffer is only 256 characters so
     successive stuffing in rapid succession may fill the buffer. }
VAR
  X : WORD;


BEGIN
  IF Exit_Vec2 = NIL THEN
    BEGIN
      Exit_Vec2 := ExitProc;          { Chain into ExitProc     }
      ExitProc := @Critical_Exit2;    { install additional exit }
      DisableInterrupts;
      GetIntVec($16, BIOS_INT16);     { Save old keyboard software ISR. }
      SetIntVec($16, @AutoTypeISR);    { Set New KeyBoard doftware ISR. }
      Enableinterrupts;
    END;
  IF SUCC(ABuffer_Head) MOD ABuffer_Size = ABuffer_Tail THEN
    Autotype := 0
  ELSE
    BEGIN
      X := 0;
      While (X < ORD(AString[0])) AND (SUCC(ABuffer_Head) MOD ABuffer_Size <> ABuffer_Tail) DO
        BEGIN
          INC(X);
          ABuffer[ABuffer_Head] := AString[X];
          ABuffer_Head := SUCC(ABuffer_Head) MOD ABuffer_Size;
        END;
      AutoType := X;
    END;
END;







Function NextKey(VAR AuxCode:CHAR):CHAR;

 { This so as to be compatable with the keyboard stuffing procedures above
    this function asks the BIOS what the next key will be and returns the
    ASCII code.  The auxillary code would be of interest if the primary code
    were 0 indicating a function key or ALT-key of some type.  This does not
    remove the key from the buffer, it just peaks ahead.
 }

BEGIN
  Inline(
{00} $B4/$01/           { MOV  AH,0001    ; Check for character service.    }
{02} $CD/$16/           { INT  $16        ; Call keyboard BIOS service.     }
{04} $74/<15-6/         { JZ   NoKey      ; If zero flag set, no key ready. }
{06} $C4/$BE/AuxCode/   { LES  DI,[AuxCode]; Get address of AuxCode.        }
{10} $26/$88/$25/       { MOV  ES:[DI],AH ; Move 2nd Code into AuxCode.     }
{13} $EB/<17-15/        { JMP  Done       ;                                 }
{15} $B0/$FF/   { Nochar: MOV  AL,255     ; Return No key ready flag.       }
{17} $88/$46/$FF);{ Done: MOV [BP -1],AL  ; Prepare for function to return AL.}
END;





Function GetKey:Char;
  { This is a good alternative to Readkey, which allows the ready access
     of function keys and other extended keys in a single char value.
     Values returned for function keys are listed as constants above. }
VAR
  CH : CHAR;
BEGIN
  While Not Keypressed DO
    INLINE($CD/$28);      {Call the DOS idle interrupt. }
  CH := Readkey;
  IF Ch = #0  THEN
    BEGIN
      Ch := ReadKey;
      IF CH < #128 THEN CH := CHR(ORD(CH) + 128);
    END;
  GetKey := CH;
END;



Procedure ClearKBDBuff;
   { Clear the BIOS keyboard buffer. }
BEGIN
  Head := Tail;
END;



Procedure CursorON;
BEGIN
    Inline($B4/$03/          { MOV   AH,03h   }
           $CD/$10/          { INT   10h      }
           $80/$E5/$5F/      { AND   CH,5Fh   ;Turn Cursor On }
           $B4/$01/          { MOV   AH,01h   }
           $CD/$10)          { INT   10h      }
END;



Procedure CursorOFF;
BEGIN
    Inline($B4/$03/          { MOV   AH,03h   }
           $CD/$10/          { INT   10h      }
           $80/$CD/$20/      { OR    CH,20h   ;Turn Cursor Off }
           $B4/$01/          { MOV   AH,01h   }
           $CD/$10)          { INT   10h      }
END;



BEGIN
  Exit_Vec2 := NIL;
END.